<?php
require_once 'config/database.php';
require_once 'includes/header.php';

// --- Determine the mode: 'dues' or 'payments' ---
$mode = $_GET['mode'] ?? 'dues';

// Fetch settings for school name
$settings = $pdo->query("SELECT setting_key, setting_value FROM settings")->fetchAll(PDO::FETCH_KEY_PAIR);
$school_name = $settings['school_name'] ?? 'Your School';

// --- Data fetching logic based on mode ---
$page_data = [];
if ($mode == 'payments') {
    // --- PAYMENT ACKNOWLEDGMENT MODE ---
    $payment_date = $_GET['payment_date'] ?? date('Y-m-d');
    $sql = "
        SELECT 
            s.student_id, s.full_name, s.guardian_contact,
            p.amount, r.receipt_id
        FROM payments p
        JOIN students s ON p.student_id = s.student_id
        JOIN receipts r ON p.payment_id = r.payment_id
        WHERE p.payment_date = ? AND p.status = 'Completed' AND r.is_deleted = 0
        ORDER BY p.created_at DESC
    ";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$payment_date]);
    $page_data = $stmt->fetchAll();

} else {
    // --- DUE REMINDER MODE (DEFAULT) ---
    $classes = $pdo->query("SELECT class_id, class_name, section FROM classes ORDER BY class_name, section")->fetchAll();
    $selected_class_id = $_GET['class_id'] ?? 'all';
    $sql = "
        SELECT 
            s.student_id, s.full_name, s.guardian_contact,
            c.class_name, c.section,
            SUM(fa.amount_due - fa.amount_paid) AS balance
        FROM students s
        JOIN fee_assignments fa ON s.student_id = fa.student_id
        JOIN classes c ON s.class_id = c.class_id
        WHERE fa.status IN ('Pending', 'Partially Paid')
    ";
    if ($selected_class_id != 'all' && is_numeric($selected_class_id)) {
        $sql .= " AND s.class_id = :class_id";
    }
    $sql .= " GROUP BY s.student_id HAVING balance > 0 ORDER BY s.full_name ASC";

    $stmt = $pdo->prepare($sql);
    if ($selected_class_id != 'all' && is_numeric($selected_class_id)) {
        $stmt->bindParam(':class_id', $selected_class_id, PDO::PARAM_INT);
    }
    $stmt->execute();
    $page_data = $stmt->fetchAll();
}
?>

<style>
/* Simple Tabs for Mode Switching */
.mode-switcher { margin-bottom: 20px; border-bottom: 2px solid #dee2e6; }
.mode-switcher a { display: inline-block; padding: 10px 20px; text-decoration: none; font-weight: 600; color: #495057; border: 2px solid transparent; }
.mode-switcher a.active { color: #007bff; border-bottom-color: #007bff; }
.card-header { display: flex; justify-content: space-between; align-items: center; }
</style>

<div class="content-header">
    <h1>WhatsApp Center</h1>
</div>

<div class="mode-switcher">
    <a href="?mode=dues" class="<?php echo ($mode == 'dues') ? 'active' : ''; ?>">Send Due Reminders</a>
    <a href="?mode=payments" class="<?php echo ($mode == 'payments') ? 'active' : ''; ?>">Send Payment Acknowledgments</a>
</div>

<div class="card">
    <!-- Filters will be rendered here -->
    <div class="card-body">
        <?php if ($mode == 'payments'): ?>
            <form method="GET" class="filter-form"><input type="hidden" name="mode" value="payments"><div class="form-group"><label for="payment_date">Show payments from date:</label><input type="date" name="payment_date" value="<?php echo htmlspecialchars($payment_date); ?>" onchange="this.form.submit()"></div></form>
        <?php else: ?>
            <form method="GET" class="filter-form"><input type="hidden" name="mode" value="dues"><div class="form-group"><label for="class_id">Filter by Class:</label><select name="class_id" onchange="this.form.submit()"><option value="all">All Classes with Dues</option><?php foreach ($classes as $class): ?><option value="<?php echo $class['class_id']; ?>" <?php echo ($selected_class_id == $class['class_id']) ? 'selected' : ''; ?>><?php echo htmlspecialchars($class['class_name'] . ' - ' . $class['section']); ?></option><?php endforeach; ?></select></div></form>
        <?php endif; ?>
    </div>
</div>

<div class="card" style="margin-top:20px;">
    <div class="card-body">
        <div class="card-header">
            <h3>
                <?php echo ($mode == 'payments') ? 'Students with Payments on ' . date("d M, Y", strtotime($payment_date)) : 'Students with Outstanding Dues'; ?>
            </h3>
            <!-- === BULK SEND BUTTONS === -->
            <?php if ($mode == 'payments' && count($page_data) > 0): ?>
                <button id="send-all-payments-btn" class="btn btn-primary">✉️ Send All Acknowledgments</button>
            <?php endif; ?>
            <?php if ($mode == 'dues' && count($page_data) > 0): ?>
                <button id="send-all-dues-btn" class="btn btn-primary">✉️ Send All Reminders</button>
            <?php endif; ?>
        </div>
        
        <table class="data-table" style="margin-top: 20px;">
            <thead>
                <tr>
                    <th>Student Name</th>
                    <th>Contact No.</th>
                    <?php if ($mode == 'payments'): ?>
                        <th>Amount Paid (₹)</th>
                    <?php else: ?>
                        <th>Class</th>
                        <th>Balance (₹)</th>
                    <?php endif; ?>
                    <th>Action</th>
                </tr>
            </thead>
            <tbody>
                <?php if (count($page_data) > 0): ?>
                    <?php foreach ($page_data as $row): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($row['full_name']); ?></td>
                            <td><?php echo htmlspecialchars($row['guardian_contact']); ?></td>
                            <?php if ($mode == 'payments'): ?>
                                <td style="text-align:right;"><?php echo number_format($row['amount'], 2); ?></td>
                                <td>
                                    <button class="btn-action btn-edit open-modal" data-student-id="<?php echo $row['student_id']; ?>" data-student-name="<?php echo htmlspecialchars($row['full_name']); ?>" data-contact="<?php echo htmlspecialchars($row['guardian_contact']); ?>" data-amount-paid="<?php echo number_format($row['amount'], 2); ?>" data-receipt-id="<?php echo $row['receipt_id']; ?>">
                                        ✉️ Send Message
                                    </button>
                                </td>
                            <?php else: ?>
                                <td><?php echo htmlspecialchars($row['class_name'] . ' - ' . $row['section']); ?></td>
                                <td style="text-align:right;"><?php echo number_format($row['balance'], 2); ?></td>
                                <td>
                                    <button class="btn-action btn-edit open-modal" data-student-id="<?php echo $row['student_id']; ?>" data-student-name="<?php echo htmlspecialchars($row['full_name']); ?>" data-contact="<?php echo htmlspecialchars($row['guardian_contact']); ?>" data-balance="<?php echo number_format($row['balance'], 2); ?>">
                                        ✉️ Send Reminder
                                    </button>
                                </td>
                            <?php endif; ?>
                        </tr>
                    <?php endforeach; ?>
                <?php else: ?>
                    <tr><td colspan="5" style="text-align:center;">No records found for this criteria.</td></tr>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- WhatsApp Message Modal -->
<div id="whatsapp-modal" class="modal">
    <div class="modal-content">
        <span class="close-button">&times;</span>
        <h3>Compose WhatsApp Message</h3>
        <p>To: <strong id="modal_student_name"></strong> (<span id="modal_contact"></span>)</p>
        <input type="hidden" id="modal_student_id">
        <div class="form-group" style="padding:0;"><label for="template_selector">Select Template</label><select id="template_selector"></select></div>
        <div class="form-group" style="padding:0;"><label for="message_template">Message:</label><textarea id="message_template" rows="8"></textarea></div>
        <div class="form-footer"><button id="generate-link-btn" class="btn btn-primary">Generate & Open WhatsApp</button></div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const modal = document.getElementById('whatsapp-modal');
    const openModalButtons = document.querySelectorAll('.open-modal');
    const closeModal = modal.querySelector('.close-button');
    const schoolName = "<?php echo addslashes($school_name); ?>";
    const templateSelector = document.getElementById('template_selector');
    const messageTextarea = document.getElementById('message_template');
    const currentMode = '<?php echo $mode; ?>';

    function generateMessage(template, data) {
        switch (template) {
            case 'due_reminder': return `Dear Parent of ${data.studentName},\n\nThis is a friendly reminder from ${schoolName}. Kindly clear the outstanding fee balance of ₹ ${data.balance} for your ward. Please note, a fine of Rupee 30 will be imposed after the 15th of the month.\n\nThank you,\n${schoolName}`;
            case 'fine_reminder': return `Dear Parent of ${data.studentName},\n\nThis is an urgent reminder from ${schoolName}. Your outstanding fee balance of ₹ ${data.balance} is overdue.\n\nPlease clear the dues immediately to avoid further late fees.\n\nThank you,\n${schoolName}`;
            case 'payment_acknowledgment': return `Dear Parent of ${data.studentName},\n\nThank you for your payment of ₹ ${data.amountPaid}. We have successfully received it.\n\nRegards,\n${schoolName}`;
            case 'custom': return '';
            default: return '';
        }
    }

    // --- Individual Message & Modal Logic (Unchanged) ---
    openModalButtons.forEach(button => {
        button.addEventListener('click', function() {
            modal.dataset.studentId = this.dataset.studentId;
            modal.dataset.studentName = this.dataset.studentName;
            modal.dataset.contact = this.dataset.contact;
            document.getElementById('modal_student_id').value = this.dataset.studentId;
            document.getElementById('modal_student_name').textContent = this.dataset.studentName;
            document.getElementById('modal_contact').textContent = this.dataset.contact;
            
            templateSelector.innerHTML = ''; let dataForMessage = {};
            if (currentMode === 'payments') {
                modal.dataset.amountPaid = this.dataset.amountPaid;
                dataForMessage = { studentName: modal.dataset.studentName, amountPaid: modal.dataset.amountPaid };
                templateSelector.innerHTML = `<option value="payment_acknowledgment">Payment Acknowledgment</option><option value="custom">Custom Message</option>`;
                templateSelector.value = 'payment_acknowledgment';
            } else {
                modal.dataset.balance = this.dataset.balance;
                dataForMessage = { studentName: modal.dataset.studentName, balance: modal.dataset.balance };
                templateSelector.innerHTML = `<option value="due_reminder">Fee Due Reminder (with Fine Warning)</option><option value="fine_reminder">Urgent Fine Reminder</option><option value="custom">Custom Message</option>`;
                templateSelector.value = 'due_reminder';
            }
            messageTextarea.value = generateMessage(templateSelector.value, dataForMessage);
            modal.style.display = 'block';
        });
    });
    templateSelector.addEventListener('change', function() {
        let dataForMessage = (currentMode === 'payments') ? { studentName: modal.dataset.studentName, amountPaid: modal.dataset.amountPaid } : { studentName: modal.dataset.studentName, balance: modal.dataset.balance };
        messageTextarea.value = generateMessage(this.value, dataForMessage);
    });
    closeModal.onclick = () => { modal.style.display = 'none'; };
    window.onclick = (event) => { if (event.target == modal) { modal.style.display = 'none'; } };
    document.getElementById('generate-link-btn').addEventListener('click', function() { /* ... existing pop-up safe logic ... */ });

    // --- BULK SEND LOGIC ---
    async function bulkSend(button, messageType, dataAttribute, template) {
        const buttons = document.querySelectorAll('tbody .open-modal');
        if (!confirm(`This will attempt to open ${buttons.length} WhatsApp tabs.\nPlease ensure your browser's pop-up blocker is disabled for this site.\n\nDo you want to proceed?`)) {
            return;
        }
        button.disabled = true;
        let count = 0;

        for (const btn of buttons) {
            count++;
            button.textContent = `Sending ${count} of ${buttons.length}...`;
            const studentData = {
                id: btn.dataset.studentId,
                name: btn.dataset.studentName,
                contact: btn.dataset.contact.replace(/[^0-9]/g, ''),
                value: btn.dataset[dataAttribute] // e.g., 'balance' or 'amountPaid'
            };
            if (studentData.contact.length === 10) { studentData.contact = '91' + studentData.contact; }
            const message = generateMessage(template, { studentName: studentData.name, [dataAttribute]: studentData.value });
            const finalUrl = `https://wa.me/${studentData.contact}?text=${encodeURIComponent(message)}`;
            
            try {
                const response = await fetch('log_whatsapp_message.php', {
                    method: 'POST', headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({ student_id: studentData.id, message_type: messageType, message_content: message, wa_link: finalUrl })
                });
                const result = await response.json();
                if (result.success) { window.open(finalUrl, '_blank'); } 
                else { console.error(`Failed to log for ${studentData.name}:`, result.error); }
            } catch (error) { console.error(`Network error for ${studentData.name}:`, error); }
            await new Promise(resolve => setTimeout(resolve, 500));
        }
        button.disabled = false;
        button.textContent = `✉️ Send All ${messageType.split(' ')[0]}s`;
        alert('Bulk sending process complete!');
    }

    const sendAllDuesBtn = document.getElementById('send-all-dues-btn');
    if (sendAllDuesBtn) {
        sendAllDuesBtn.addEventListener('click', () => bulkSend(sendAllDuesBtn, 'Due Reminder (Bulk)', 'balance', 'due_reminder'));
    }

    const sendAllPaymentsBtn = document.getElementById('send-all-payments-btn');
    if (sendAllPaymentsBtn) {
        sendAllPaymentsBtn.addEventListener('click', () => bulkSend(sendAllPaymentsBtn, 'Payment Acknowledgment (Bulk)', 'amountPaid', 'payment_acknowledgment'));
    }
});
</script>

<?php require_once 'includes/footer.php'; ?>

