<?php
require_once 'config/database.php';

// --- Robust Session Handling ---
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST' || !isset($_POST['action'])) {
    header('Location: receipts.php');
    exit();
}

// --- Security Check: Verify User is Logged In ---
if (!isset($_SESSION['user_id'])) {
    header("Location: receipts.php?error=Session expired. Please log in again.");
    exit();
}

$action = $_POST['action'];
$logged_in_user_id = $_SESSION['user_id'];

if ($action == 'delete') {
    if (empty($_POST['receipt_id']) || empty(trim($_POST['deletion_reason']))) {
        header("Location: receipts.php?error=Receipt ID or deletion reason was missing.");
        exit();
    }
    
    $receipt_id = $_POST['receipt_id'];
    $reason = trim($_POST['deletion_reason']);

    try {
        // --- 1. Begin Database Transaction for Safety ---
        $pdo->beginTransaction();

        // --- 2. Fetch receipt details, including student_id directly from the table ---
        $stmt = $pdo->prepare("SELECT payment_id, student_id, receipt_data FROM receipts WHERE receipt_id = ?");
        $stmt->execute([$receipt_id]);
        $receipt = $stmt->fetch();

        if (!$receipt) {
            throw new Exception("Receipt not found. It may have already been deleted.");
        }
        $payment_id = $receipt['payment_id'];
        $student_id = $receipt['student_id']; // <-- The reliable student_id
        $receipt_data = json_decode($receipt['receipt_data'], true);

        // This check is now more robust.
        if (!$student_id) {
            throw new Exception("Student ID is missing from the receipt record itself.");
        }

        // --- 3. Reverse the Financial Impact ---
        // Re-credit the amount to the student's dues for each item on the receipt
        if (isset($receipt_data['details']) && is_array($receipt_data['details'])) {
            foreach ($receipt_data['details'] as $item) {
                $title = $item['title'];
                // Skip reversing fines as they are virtual and don't exist in the assignments table
                if (strpos($title, 'Fine for: ') === 0) {
                    continue; 
                }
                
                $stmt = $pdo->prepare("UPDATE fee_assignments SET amount_paid = amount_paid - ?, status = IF(amount_paid <= 0, 'Pending', 'Partially Paid') WHERE assignment_title = ? AND student_id = ?");
                $stmt->execute([$item['amount_paid'], $title, $student_id]);
            }
        }
        
        // --- 4. Archive Payment and Receipt (Robust Method) ---
        //  a) Copy the payment record to the archive table by explicitly naming columns
        $sql_archive_payment = "
            INSERT INTO deleted_student_payments (payment_id, student_id, amount, payment_mode, ref_no, payment_date, remarks, collected_by, created_at, status, archived_at)
            SELECT payment_id, student_id, amount, payment_mode, ref_no, payment_date, remarks, collected_by, created_at, status, NOW()
            FROM payments WHERE payment_id = ?";
        $pdo->prepare($sql_archive_payment)->execute([$payment_id]);
        
        //  b) Copy the receipt record to the archive table by explicitly naming columns
        $sql_archive_receipt = "
            INSERT INTO deleted_student_receipts (receipt_id, payment_id, student_id, receipt_data, created_by, created_at, is_deleted, deletion_reason, archived_at, deleted_by_user_id)
            SELECT receipt_id, payment_id, student_id, receipt_data, created_by, created_at, is_deleted, ?, NOW(), ?
            FROM receipts WHERE receipt_id = ?";
        $pdo->prepare($sql_archive_receipt)->execute([$reason, $logged_in_user_id, $receipt_id]);


        // --- 5. Permanently Delete the Originals from Active Tables ---
        $pdo->prepare("DELETE FROM payments WHERE payment_id = ?")->execute([$payment_id]);
        $pdo->prepare("DELETE FROM receipts WHERE receipt_id = ?")->execute([$receipt_id]);
        
        // --- 6. Finalize the Transaction ---
        $pdo->commit();

        header("Location: receipts.php?success=Receipt and payment have been successfully deleted and archived.");

    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        header("Location: receipts.php?error=Operation failed: " . $e->getMessage());
    }
    exit();
}
?>

