<?php
// Always set the content type to JSON for API-like scripts
header('Content-Type: application/json');

// --- Robust Session Handling ---
// It's crucial for scripts called by JavaScript to manage their own session
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Include the database connection AFTER starting the session.
require_once 'config/database.php';

// --- Security Check: Verify User is Logged In ---
// If the user's session has expired, this is the most common point of failure.
if (!isset($_SESSION['user_id'])) {
    // Send a clear JSON error back to the JavaScript and stop execution.
    // This prevents a fatal error and the generic "network error" message.
    echo json_encode(['success' => false, 'error' => 'Your session has expired. Please log out and log back in.']);
    exit();
}

// --- Main Logic ---
try {
    // Get the posted data sent from the JavaScript fetch request
    $post_data = json_decode(file_get_contents('php://input'), true);

    // Validate that the incoming data is valid JSON and has the required fields
    if (json_last_error() !== JSON_ERROR_NONE || empty($post_data['student_id']) || !isset($post_data['message_content'])) {
        throw new Exception("Invalid or incomplete data received from the form.");
    }
    
    // Get the logged-in user's ID from the now-verified session
    $user_id = $_SESSION['user_id'];

    $sql = "INSERT INTO whatsapp_logs (user_id, student_id, message_type, message_content, wa_link) VALUES (?, ?, ?, ?, ?)";
    $stmt = $pdo->prepare($sql);
    
    $stmt->execute([
        $user_id,
        $post_data['student_id'],
        $post_data['message_type'] ?? 'Custom',
        $post_data['message_content'],
        $post_data['wa_link'] ?? ''
    ]);

    // If the insert is successful, send a success response
    echo json_encode(['success' => true]);

} catch (Exception $e) {
    // If any other error occurs during the process, send a structured JSON error
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}
?>

