<?php
header('Content-Type: application/json');
require_once 'config/database.php';

$student_id = $_GET['id'] ?? 0;
if (!$student_id) {
    echo json_encode(['error' => 'Invalid Student ID']);
    exit();
}

try {
    // 1. Fetch all active fine rules
    $fine_rules_raw = $pdo->query("SELECT * FROM fines WHERE is_active = 1")->fetchAll();
    $fine_rules = [];
    foreach ($fine_rules_raw as $rule) {
        $fine_rules[$rule['fee_head_id']] = $rule;
    }

    // 2. Fetch student info
    $student = $pdo->prepare("SELECT * FROM students WHERE student_id = ?");
    $student->execute([$student_id]);
    $student_info = $student->fetch();

    // 3. Fetch ONLY PENDING fees
    $stmt = $pdo->prepare("
        SELECT fa.assignment_id, fa.assignment_title, fa.amount_due, fa.amount_paid, fa.due_date, fs.fee_head_id
        FROM fee_assignments fa
        JOIN fee_structures fs ON fa.structure_id = fs.structure_id
        WHERE fa.student_id = ? AND fa.status = 'Pending'
        ORDER BY fa.due_date ASC
    ");
    $stmt->execute([$student_id]);
    $dues = $stmt->fetchAll();
    
    // 4. Calculate fines with robust date logic
    $today = new DateTime(); 
    $today->setTime(0, 0, 0); // Normalize to the start of the day
    
    $all_dues_with_fines = $dues;

    foreach ($dues as $due) {
        $due_date = new DateTime($due['due_date']);
        $due_date->setTime(0, 0, 0); // Normalize to the start of the day

        // Skip if the due date is in the future
        if ($today <= $due_date) {
            continue;
        }
        
        // Check if a fine rule exists for this fee's type
        if (isset($fine_rules[$due['fee_head_id']])) {
            $rule = $fine_rules[$due['fee_head_id']];
            
            // Calculate the date when the fine should start applying
            $fine_start_date = clone $due_date;
            $fine_start_date->modify('+' . ($rule['start_after_days'] + 1) . ' days');

            // Apply fine only if today is past the fine start date
            if ($today >= $fine_start_date) {
                $fine_amount = 0;
                
                if ($rule['recurring_period'] == 'Daily') {
                    // It's a recurring fine. Calculate for each day from the start date.
                    $chargeable_days = $today->diff($fine_start_date)->days + 1;
                    $fine_amount = $chargeable_days * (float)$rule['fine_value'];
                } else {
                    // It's a one-time, fixed fine.
                    $fine_amount = (float)$rule['fine_value'];
                }

                if ($rule['fine_type'] == 'Percentage') {
                    // Note: Percentage calculation for recurring fines can be complex.
                    // This implementation assumes a percentage of the original due amount per day.
                    $fine_amount = ((float)$due['amount_due'] * $fine_amount) / 100;
                }

                if ($fine_amount > 0) {
                    $all_dues_with_fines[] = [
                        'assignment_id' => 'fine_' . $due['assignment_id'],
                        'assignment_title' => 'Fine for: ' . $due['assignment_title'],
                        'amount_due' => $fine_amount,
                        'amount_paid' => 0,
                        'due_date' => date('Y-m-d')
                    ];
                }
            }
        }
    }

    echo json_encode(['student' => $student_info, 'dues' => $all_dues_with_fines]);

} catch (PDOException $e) {
    echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
}
?>

