<?php
require_once 'config/database.php';
require_once 'includes/header.php';

// --- Process search filters ---
$search_term = $_GET['search_term'] ?? '';
$start_date = $_GET['start_date'] ?? '';
$end_date = $_GET['end_date'] ?? '';

$receipts = [];
$params = [];

// Base SQL query to join receipts with payments and students
$sql = "
    SELECT 
        r.receipt_id,
        p.payment_date,
        p.amount,
        s.student_uid,
        s.full_name
    FROM receipts r
    JOIN payments p ON r.payment_id = p.payment_id
    JOIN students s ON r.student_id = s.student_id
    WHERE r.is_deleted = 0 
";

// Dynamically add conditions to the query based on user-provided filters
if (!empty($search_term)) {
    // Allows searching by name, UID, or the exact receipt ID number
    $sql .= " AND (s.full_name LIKE ? OR s.student_uid LIKE ? OR r.receipt_id = ?)";
    $like_term = "%{$search_term}%";
    array_push($params, $like_term, $like_term, $search_term);
}
if (!empty($start_date)) {
    $sql .= " AND p.payment_date >= ?";
    $params[] = $start_date;
}
if (!empty($end_date)) {
    $sql .= " AND p.payment_date <= ?";
    $params[] = $end_date;
}

// Order by the most recent receipts first and limit results for performance
$sql .= " ORDER BY r.created_at DESC LIMIT 100";

try {
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $receipts = $stmt->fetchAll();
} catch (PDOException $e) {
    echo "<div class='alert alert-danger'>Database Error: " . $e->getMessage() . "</div>";
}
?>

<div class="content-header">
    <h1>Search Receipts</h1>
    <p>Find, reprint, or delete any past receipt.</p>
</div>

<!-- Message Display Area -->
<?php if(isset($_GET['success'])): ?><div class="alert alert-success"><?php echo htmlspecialchars($_GET['success']); ?></div><?php endif; ?>
<?php if(isset($_GET['error'])): ?><div class="alert alert-danger"><?php echo htmlspecialchars($_GET['error']); ?></div><?php endif; ?>


<div class="card">
    <div class="card-body">
        <form method="GET" action="receipts.php" class="filter-form">
            <div class="form-group" style="flex-grow: 2;">
                <label for="search_term">Search by Receipt #, UID, or Name</label>
                <input type="text" name="search_term" id="search_term" value="<?php echo htmlspecialchars($search_term); ?>" placeholder="e.g., 123 or S2025001">
            </div>
            <div class="form-group">
                <label for="start_date">From Date</label>
                <input type="date" name="start_date" id="start_date" value="<?php echo htmlspecialchars($start_date); ?>">
            </div>
            <div class="form-group">
                <label for="end_date">To Date</label>
                <input type="date" name="end_date" id="end_date" value="<?php echo htmlspecialchars($end_date); ?>">
            </div>
            <button type="submit" class="btn btn-primary">Search</button>
        </form>
    </div>
</div>

<div class="card" style="margin-top:20px;">
    <div class="card-body">
        <h3>Search Results</h3>
        <table class="data-table">
            <thead>
                <tr>
                    <th>Receipt No.</th>
                    <th>Date</th>
                    <th>Student Name</th>
                    <th>Amount (₹)</th>
                    <th>Action</th>
                </tr>
            </thead>
            <tbody>
                <?php if (count($receipts) > 0): ?>
                    <?php foreach ($receipts as $receipt): ?>
                        <tr>
                            <td>RCPT-<?php echo str_pad($receipt['receipt_id'], 6, '0', STR_PAD_LEFT); ?></td>
                            <td><?php echo date("d-m-Y", strtotime($receipt['payment_date'])); ?></td>
                            <td><?php echo htmlspecialchars($receipt['full_name']); ?></td>
                            <td style="text-align:right;"><?php echo number_format($receipt['amount'], 2); ?></td>
                            <td>
                                <a href="view_receipt.php?id=<?php echo $receipt['receipt_id']; ?>" class="btn-action btn-edit" target="_blank">View/Print</a>
                                <!-- === ADDED DELETE BUTTON === -->
                                <button class="btn-action btn-delete open-modal" data-receipt-id="<?php echo $receipt['receipt_id']; ?>">Delete</button>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php else: ?>
                    <tr>
                        <td colspan="5" style="text-align:center;">No receipts found matching your criteria.</td>
                    </tr>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- === ADDED: Deletion Modal === -->
<div id="delete-modal" class="modal">
    <div class="modal-content">
        <span class="close-button">&times;</span>
        <h3>Delete Receipt</h3>
        <p>You must provide a reason for deleting this receipt. This action will be logged.</p>
        <form action="process_receipt.php" method="POST">
            <input type="hidden" name="action" value="delete">
            <input type="hidden" name="receipt_id" id="modal_receipt_id">
            <div class="form-group" style="padding: 0;">
                <label for="deletion_reason">Reason for Deletion:</label>
                <textarea name="deletion_reason" id="deletion_reason" rows="4" required></textarea>
            </div>
            <div class="form-footer">
                <button type="submit" class="btn btn-delete">Confirm Deletion</button>
            </div>
        </form>
    </div>
</div>

<!-- === ADDED: CSS for Modal === -->
<style>
.modal {
    display: none; /* Hidden by default */
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    overflow: auto;
    background-color: rgba(0,0,0,0.5);
}

.modal-content {
    background-color: #fefefe;
    margin: 15% auto;
    padding: 25px;
    border: 1px solid #888;
    width: 80%;
    max-width: 500px;
    border-radius: 8px;
    position: relative;
}

.close-button {
    color: #aaa;
    float: right;
    font-size: 28px;
    font-weight: bold;
    position: absolute;
    top: 10px;
    right: 20px;
}

.close-button:hover,
.close-button:focus {
    color: black;
    text-decoration: none;
    cursor: pointer;
}
</style>

<script>
// --- Modal Control JavaScript ---
const modal = document.getElementById('delete-modal');
const closeButton = document.querySelector('.close-button');
const openModalButtons = document.querySelectorAll('.open-modal');
const modalReceiptIdInput = document.getElementById('modal_receipt_id');

openModalButtons.forEach(button => {
    button.addEventListener('click', function() {
        const receiptId = this.dataset.receiptId;
        modalReceiptIdInput.value = receiptId;
        modal.style.display = 'block';
    });
});

closeButton.addEventListener('click', () => modal.style.display = 'none');
window.addEventListener('click', (event) => {
    if (event.target == modal) {
        modal.style.display = 'none';
    }
});
</script>


<?php
require_once 'includes/footer.php';
?>

