<?php
require_once 'config/database.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: fee_collection.php');
    exit();
}

// --- 1. Data Validation & Sanitization ---
if (empty($_POST['student_id']) || empty($_POST['assignment_ids']) || !isset($_POST['amount_paid']) || empty($_POST['payment_date'])) {
    header("Location: fee_collection.php?error=Missing required payment data.");
    exit();
}

$student_id = $_POST['student_id'];
$assignment_ids_from_form = $_POST['assignment_ids'];
$amount_to_apply_total = (float)$_POST['amount_paid'];
$payment_date = $_POST['payment_date'];
$payment_mode = $_POST['payment_mode'] ?? 'Cash';
$ref_no = trim($_POST['ref_no'] ?? '');
$collected_by = $_SESSION['user_id'];

if ($amount_to_apply_total <= 0) {
    header("Location: fee_collection.php?error=Payment amount must be greater than zero.");
    exit();
}

try {
    // --- 2. Begin Database Transaction ---
    $pdo->beginTransaction();

    // --- 3. Insert the master payment record ---
    $sql_payment = "INSERT INTO payments (student_id, amount, payment_mode, ref_no, payment_date, collected_by) VALUES (?, ?, ?, ?, ?, ?)";
    $stmt_payment = $pdo->prepare($sql_payment);
    $stmt_payment->execute([$student_id, $amount_to_apply_total, $payment_mode, $ref_no, $payment_date, $collected_by]);
    $payment_id = $pdo->lastInsertId();

    // --- 4. Reconstruct and Validate all items being paid (dues and fines) ---
    $real_assignment_ids = [];
    $fine_generating_assignment_ids = [];
    foreach ($assignment_ids_from_form as $id) {
        if (strpos($id, 'fine_') === 0) {
            $fine_generating_assignment_ids[] = (int)str_replace('fine_', '', $id);
        } else {
            $real_assignment_ids[] = (int)$id;
        }
    }

    $items_to_pay = [];

    // A) Get details for the real dues being paid
    if (!empty($real_assignment_ids)) {
        $placeholders = implode(',', array_fill(0, count($real_assignment_ids), '?'));
        $sql_dues = "SELECT * FROM fee_assignments WHERE assignment_id IN ($placeholders) AND student_id = ? ORDER BY due_date ASC";
        $stmt_dues = $pdo->prepare($sql_dues);
        $stmt_dues->execute(array_merge($real_assignment_ids, [$student_id]));
        $items_to_pay = array_merge($items_to_pay, $stmt_dues->fetchAll());
    }

    // B) Server-side recalculation of fines for security
    if (!empty($fine_generating_assignment_ids)) {
        $fine_rules_raw = $pdo->query("SELECT * FROM fines WHERE is_active = 1")->fetchAll();
        $fine_rules = [];
        foreach ($fine_rules_raw as $rule) { $fine_rules[$rule['fee_head_id']] = $rule; }

        $placeholders = implode(',', array_fill(0, count($fine_generating_assignment_ids), '?'));
        $sql_fine_dues = "SELECT fa.*, fs.fee_head_id FROM fee_assignments fa JOIN fee_structures fs ON fa.structure_id = fs.structure_id WHERE fa.assignment_id IN ($placeholders) AND fa.student_id = ?";
        $stmt_fine_dues = $pdo->prepare($sql_fine_dues);
        $stmt_fine_dues->execute(array_merge($fine_generating_assignment_ids, [$student_id]));
        $dues_generating_fines = $stmt_fine_dues->fetchAll();

        $today = new DateTime();
        foreach ($dues_generating_fines as $due) {
            if (isset($fine_rules[$due['fee_head_id']])) {
                $rule = $fine_rules[$due['fee_head_id']];
                $due_date = new DateTime($due['due_date']);
                $days_overdue = $today->diff($due_date)->days;

                if ($days_overdue > $rule['start_after_days']) {
                    $fine_amount = 0;
                    
                    if ($rule['recurring_period'] == 'Daily') {
                        $billable_late_days = $days_overdue - $rule['start_after_days'];
                        $base_fine = ($rule['fine_type'] == 'Fixed') ? $rule['fine_value'] : (($due['amount_due'] * $rule['fine_value']) / 100);
                        $fine_amount = $base_fine * $billable_late_days;
                    } else { // 'None' - One time fine
                        $fine_amount = ($rule['fine_type'] == 'Fixed') ? $rule['fine_value'] : (($due['amount_due'] * $rule['fine_value']) / 100);
                    }

                    if ($fine_amount > 0) {
                         $items_to_pay[] = [ 'is_fine' => true, 'assignment_title' => 'Fine for: ' . $due['assignment_title'], 'amount_due' => round($fine_amount, 2), 'amount_paid' => 0, 'assignment_id' => $due['assignment_id'] ];
                    }
                }
            }
        }
    }

    // --- 5. Distribute the payment amount and update database ---
    $receipt_details = [];
    $amount_to_apply = $amount_to_apply_total;

    foreach ($items_to_pay as $item) {
        if ($amount_to_apply <= 0) break;

        $balance_on_this_item = (float)$item['amount_due'] - (float)$item['amount_paid'];
        $payment_for_this_item = min($amount_to_apply, $balance_on_this_item);
        
        if (!isset($item['is_fine'])) {
            $new_amount_paid = (float)$item['amount_paid'] + $payment_for_this_item;
            $new_status = ($new_amount_paid >= $item['amount_due']) ? 'Paid' : 'Partially Paid';
            
            $stmt_update = $pdo->prepare("UPDATE fee_assignments SET amount_paid = ?, status = ? WHERE assignment_id = ?");
            $stmt_update->execute([$new_amount_paid, $new_status, $item['assignment_id']]);
        }
        
        $amount_to_apply -= $payment_for_this_item;
        
        $receipt_details[] = [ 'assignment_id' => $item['assignment_id'], 'title' => $item['assignment_title'], 'amount_paid' => $payment_for_this_item ];
    }

    // --- 6. Calculate new remaining balance ---
    $stmt_balance = $pdo->prepare("SELECT SUM(amount_due - amount_paid) FROM fee_assignments WHERE student_id = ? AND status IN ('Pending', 'Partially Paid')");
    $stmt_balance->execute([$student_id]);
    $remaining_balance = $stmt_balance->fetchColumn() ?? 0;

    // --- 7. Generate and store receipt data ---
    $student_info = $pdo->query("SELECT full_name, student_uid FROM students WHERE student_id = $student_id")->fetch();
    $collector_info = $pdo->query("SELECT full_name FROM users WHERE user_id = $collected_by")->fetch();
    
    $receipt_data = json_encode([
        'receipt_no' => 'RCPT-' . str_pad($payment_id, 6, '0', STR_PAD_LEFT),
        'payment_id' => $payment_id, 'date' => $payment_date, 'student_name' => $student_info['full_name'], 'student_uid' => $student_info['student_uid'],
        'payment_mode' => $payment_mode, 'ref_no' => $ref_no, 'total_paid' => $amount_to_apply_total, 'details' => $receipt_details,
        'collected_by_name' => $collector_info['full_name'] ?? 'N/A', 'remaining_balance' => $remaining_balance
    ]);

    $stmt_receipt = $pdo->prepare("INSERT INTO receipts (payment_id, student_id, receipt_data, created_by) VALUES (?, ?, ?, ?)");
    $stmt_receipt->execute([$payment_id, $student_id, $receipt_data, $collected_by]);
    $receipt_id = $pdo->lastInsertId();

    // --- 8. Commit the transaction ---
    $pdo->commit();

    // --- 9. Redirect to receipt page ---
    header("Location: view_receipt.php?id=" . $receipt_id);
    exit();

} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    header("Location: fee_collection.php?error=" . urlencode("Transaction failed: " . $e->getMessage()));
    exit();
}
?>

