<?php
require_once 'config/database.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: bulk_fee_assignment.php');
    exit();
}

// --- 1. Validate Inputs ---
if (empty($_POST['class_id']) || empty($_POST['months']) || empty($_POST['assignment_title']) || empty($_POST['due_day'])) {
    header("Location: bulk_fee_assignment.php?error=All fields are required.");
    exit();
}

$class_id = $_POST['class_id'];
$selected_months = $_POST['months']; // Array of month numbers (e.g., ['04', '05'])
$base_title = trim($_POST['assignment_title']);
$due_day = (int)$_POST['due_day'];

try {
    // --- 2. Fetch necessary data ---
    // Get the academic session year
    $session_str = $pdo->query("SELECT setting_value FROM settings WHERE setting_key = 'academic_session'")->fetchColumn();
    list($start_year, $end_year) = explode('-', $session_str);

    // Get the fee structure for the selected class
    $structure_stmt = $pdo->prepare("SELECT * FROM fee_structures WHERE class_id = ?");
    $structure_stmt->execute([$class_id]);
    $structure_items = $structure_stmt->fetchAll();

    if (empty($structure_items)) {
        throw new Exception("No fee structure is defined for this class.");
    }

    // Get all active students in the selected class
    $student_stmt = $pdo->prepare("SELECT student_id FROM students WHERE class_id = ? AND is_active = 1");
    $student_stmt->execute([$class_id]);
    $students = $student_stmt->fetchAll(PDO::FETCH_COLUMN);

    if (empty($students)) {
        throw new Exception("No active students found in this class.");
    }

    // --- 3. Begin Database Transaction ---
    $pdo->beginTransaction();

    // Prepare statement for insertion
    $insert_sql = "INSERT INTO fee_assignments (student_id, structure_id, assignment_title, amount_due, due_date) VALUES (?, ?, ?, ?, ?)";
    $insert_stmt = $pdo->prepare($insert_sql);
    
    // Prepare statement for checking duplicates
    $check_sql = "SELECT COUNT(*) FROM fee_assignments WHERE student_id = ? AND assignment_title = ?";
    $check_stmt = $pdo->prepare($check_sql);

    // --- 4. Main Loop ---
    foreach ($selected_months as $month_num) {
        // Determine the correct year for the month based on the academic session
        $year = ($month_num >= '04' && $month_num <= '12') ? $start_year : $end_year;
        $month_name = date("F", mktime(0, 0, 0, $month_num, 10));
        
        $assignment_title = "$base_title - $month_name $year";
        $due_date = "$year-$month_num-" . str_pad($due_day, 2, '0', STR_PAD_LEFT);

        foreach ($students as $student_id) {
            // Check for duplicates before inserting
            $check_stmt->execute([$student_id, $assignment_title]);
            if ($check_stmt->fetchColumn() > 0) {
                continue; // Skip if this exact fee title is already assigned to this student
            }

            foreach ($structure_items as $item) {
                $insert_stmt->execute([
                    $student_id,
                    $item['structure_id'],
                    $assignment_title,
                    $item['amount'],
                    $due_date
                ]);
            }
        }
    }

    // --- 5. Commit Transaction ---
    $pdo->commit();
    header("Location: bulk_fee_assignment.php?success=Fees successfully generated for " . count($selected_months) . " month(s).");

} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    header("Location: bulk_fee_assignment.php?error=" . urlencode("An error occurred: " . $e->getMessage()));
}
exit();
?>
