<?php
require_once 'config/database.php';
require_once 'includes/header.php';

// Validate Class ID from URL
if (!isset($_GET['class_id']) || !is_numeric($_GET['class_id'])) {
    header("Location: fee_structures.php?error=Invalid class selected.");
    exit();
}
$class_id = $_GET['class_id'];

try {
    // 1. Fetch class details
    $stmt = $pdo->prepare("SELECT * FROM classes WHERE class_id = ?");
    $stmt->execute([$class_id]);
    $class = $stmt->fetch();
    if (!$class) {
        header("Location: fee_structures.php?error=Class not found.");
        exit();
    }

    // 2. Fetch existing fee structure for this class
    $stmt = $pdo->prepare("
        SELECT fs.structure_id, fh.head_name, fs.amount
        FROM fee_structures fs
        JOIN fee_heads fh ON fs.fee_head_id = fh.fee_head_id
        WHERE fs.class_id = ?
        ORDER BY fh.head_name
    ");
    $stmt->execute([$class_id]);
    $existing_structure = $stmt->fetchAll();

    // 3. Fetch fee heads that are NOT already in this class's structure
    $stmt = $pdo->prepare("
        SELECT fee_head_id, head_name FROM fee_heads
        WHERE fee_head_id NOT IN (SELECT fee_head_id FROM fee_structures WHERE class_id = ?)
        ORDER BY head_name
    ");
    $stmt->execute([$class_id]);
    $available_heads = $stmt->fetchAll();

} catch (PDOException $e) {
    die("Database error: " . $e->getMessage());
}
?>

<div class="content-header">
    <h1>Manage Fee Structure for: <?php echo htmlspecialchars($class['class_name'] . ' - ' . $class['section']); ?></h1>
    <a href="fee_structures.php" class="btn">⬅️ Back to Class List</a>
</div>

<?php if(isset($_GET['success'])): ?><div class="alert alert-success"><?php echo htmlspecialchars($_GET['success']); ?></div><?php endif; ?>
<?php if(isset($_GET['error'])): ?><div class="alert alert-danger"><?php echo htmlspecialchars($_GET['error']); ?></div><?php endif; ?>

<div class="card">
    <div class="card-body">
        <h3>Existing Structure</h3>
        <table class="data-table">
            <thead><tr><th>Fee Head</th><th>Amount (₹)</th><th>Action</th></tr></thead>
            <tbody>
                <?php if ($existing_structure): ?>
                    <?php foreach ($existing_structure as $item): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($item['head_name']); ?></td>
                        <td><?php echo number_format($item['amount'], 2); ?></td>
                        <td>
                            <form action="process_fee_structure.php" method="POST" onsubmit="return confirm('Are you sure you want to remove this?');">
                                <input type="hidden" name="action" value="delete">
                                <input type="hidden" name="structure_id" value="<?php echo $item['structure_id']; ?>">
                                <input type="hidden" name="class_id" value="<?php echo $class_id; // For redirecting back ?>">
                                <button type="submit" class="btn-action btn-delete">Remove</button>
                            </form>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                <?php else: ?>
                    <tr><td colspan="3" style="text-align:center;">No fee structure defined for this class yet.</td></tr>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<div class="card" style="margin-top: 20px;">
    <div class="card-body">
        <h3>Add New Fee Component</h3>
        <form action="process_fee_structure.php" method="POST">
            <input type="hidden" name="action" value="add">
            <input type="hidden" name="class_id" value="<?php echo $class_id; ?>">
            <div class="form-row">
                <div class="form-group">
                    <label for="fee_head_id">Fee Head</label>
                    <select id="fee_head_id" name="fee_head_id" required>
                        <option value="">-- Select Fee Head --</option>
                        <?php foreach($available_heads as $head): ?>
                        <option value="<?php echo $head['fee_head_id']; ?>"><?php echo htmlspecialchars($head['head_name']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group">
                    <label for="amount">Amount (₹)</label>
                    <input type="number" step="0.01" id="amount" name="amount" required>
                </div>
            </div>
            <div class="form-footer">
                <button type="submit" class="btn btn-primary">Add to Structure</button>
            </div>
        </form>
    </div>
</div>


<?php
require_once 'includes/footer.php';
?>